package gov.va.med.domain.service.messaging.decode;

import gov.va.med.domain.model.IPayload;
import gov.va.med.domain.service.messaging.MessagingConstants;
import gov.va.med.domain.service.messaging.MessagingException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

/**
 * MessageDecoder
 *
 * @author Odysseas Pentakalos
 * @version $Id: MessageDecoder.java,v 1.9 2005/08/03 20:18:08 joel.goldberg Exp $
 * @since MHV 2.0 <br>Mar 1, 2005
 */
public abstract class MessageDecoder implements MessagingConstants
{
    /**
     * 
     * @param originalPayload - not generally used
     * @param encodedPayload - the incoming message usually a String 
     * @return the object built from the encodedPayload.   
     * @throws MessagingException
     * @deprecated - use decode(Object encodedPayload) that omits originalPayload
     */
    public IPayload decode(IPayload originalPayload, Object encodedPayload)
			throws MessagingException {
		return decode(encodedPayload);
	}
    
	/**
	 * This method must be implemented to create a first class business object
	 * from the values found in the encodedPayload. The return must implement
	 * IPayload.
	 * 
	 * @param encodedPayload - the incoming message usually a String
	 * @return the object built from the encodedPayload.
	 * @throws MessagingException
	 */
    public abstract IPayload decode(Object encodedPayload) 
    	throws MessagingException;
	
	/**
	 * Tries to apply different data formatters to parser the string to a valid Date.  
	 * If none work, a MessagingException is thrown <P>
	 * Currently supported formats:
	 * <UL>
	 * <LI>	yyyyMMdd
	 * <LI>	yyyyMMddHHmm
	 * <LI>	yyyyMMddHHmmss
	 * <LI>	yyyyMMddHHmmss.SSS
	 * <LI>	yyyyMMddHHmmss.SSSZ (e.g. 20040328134602.1234+0600 - NOTE truncation to 3 decimal places)
	 * </UL>
	 * 
	 * @param anyFormatString
	 * @return the resulting Date
	 * @throws MessagingException if anyFormatString cannot be parsed into either a Date or a null 
	 */
	public static Date stringToDate(String anyFormatString)
			throws MessagingException {

		if (anyFormatString == null || anyFormatString.trim().length() == 0) {
			return null;
		}
		anyFormatString = removeTimeZone(anyFormatString);
	    if (anyFormatString.trim().length() == 0)
	        return null;
		try {
		    if (anyFormatString.length() == 8) {
				return yyyyMMddToDate(anyFormatString);
			}
			if (anyFormatString.length() == 12) {
				return yyyyMMddHHmmToDate(anyFormatString);
			}
			if (anyFormatString.length() == 14) {
				return yyyyMMddHHmmssToDate(anyFormatString);
			}
			if (anyFormatString.length() == 19) {
			    return longFormatToDate(anyFormatString);
			}			
			if (anyFormatString.length() > 19) {
				return longFormatWithZoneToDate(anyFormatString);
			}
		} catch (ParseException e) {
			throw new MessagingException("Malformed date [" + anyFormatString
					+ "]in message.", e);
		}
		throw new MessagingException("Malformed date [" + anyFormatString
				+ "]in message.");
	}
	
	protected static Date yyyyMMddToDate(String yyyyMMdd) throws ParseException {
		// note new SimpleDateFormat created because it is not thread-safe.
		return new SimpleDateFormat(yyyyMMddFormat).parse(yyyyMMdd);
	}	
	
	protected static Date yyyyMMddHHmmToDate(String yyyyMMddHHmm)
			throws ParseException {
		// note new SimpleDateFormat created because it is not thread-safe.
		return new SimpleDateFormat(yyyyMMddHHmmFormat).parse(yyyyMMddHHmm);
	}  
	
	protected static Date yyyyMMddHHmmssToDate(String yyyyMMddHHmmss)
			throws ParseException {
		// note new SimpleDateFormat created because it is not thread-safe.
		return new SimpleDateFormat(yyyyMMddHHmmssFormat).parse(yyyyMMddHHmmss);
	}  
	
	protected static Date longFormatToDate(String longFormat)
			throws ParseException {
		//Time truncated to 3 decimal places.
		longFormat = truncateToMilliseconds(longFormat);
		
		// note new SimpleDateFormat created because it is not thread-safe.
		SimpleDateFormat sdf = new SimpleDateFormat(yyyyMMddHHmmssSSSFormat);
		Date date = sdf.parse(longFormat);
		return date;
	}  	
	
	protected static Date longFormatWithZoneToDate(String longFormatWithZone)
			throws ParseException {
		//Time truncated to 3 decimal places.
		longFormatWithZone = truncateToMilliseconds(longFormatWithZone);

		// note new SimpleDateFormat created because it is not thread-safe.
		SimpleDateFormat sdf = new SimpleDateFormat(yyyyMMddHHmmssSSSZFormat);
		Date date = sdf.parse(longFormatWithZone);
		return date;
	}  
	
	/**
	 * Removes timezone info if present.
	 * 
	 * 
	 * @param longFormatWithZone
	 * @return longFormat
	 */
	private static String removeTimeZone(String longFormatWithZone)
	{
	    int zoneSignPos = longFormatWithZone.indexOf("-");
	    //try "+" if "-" is not found.
	    if (zoneSignPos < 0)
	        zoneSignPos = longFormatWithZone.indexOf("+");
	    if(zoneSignPos > -1)
	    {
	        return longFormatWithZone.substring(0, zoneSignPos);
	    }
	    return longFormatWithZone;
	}

	/**
	 * Truncate to 3 decimal places. Truncate only if a decimal point is present.
	 * 
	 * @param longFormat the date string in long format
	 * @return the truncated date string 
	 */
	private static String truncateToMilliseconds(String longFormat) {
	    int pos = longFormat.lastIndexOf(".");
		
		if (pos > 0) {
			String string1 = longFormat.substring(0, pos+4);			
			String string2 = longFormat.substring(pos+5);
			longFormat = string1 + string2;
		}
		return longFormat;
	}
	
	public Integer stringToInteger(String anIntString) 
    {
        if (anIntString==null ||  anIntString.trim().length()==0  ) {
          return null; 
        }
        return new Integer(anIntString);  
    }  
	

}
